// Admin UI interactions (no dependencies)
// - Mobile sidebar drawer
// - Theme toggle (persisted)
// - Search shortcut
// - Copy-to-clipboard helpers + toasts

(function () {
  const $ = (sel, root = document) => root.querySelector(sel);
  const $$ = (sel, root = document) => Array.from(root.querySelectorAll(sel));

  function prefersDark() {
    return window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches;
  }

  function getTheme() {
    try {
      const t = localStorage.getItem('admin:theme');
      return t === 'dark' || t === 'light' ? t : null;
    } catch {
      return null;
    }
  }

  function setTheme(theme) {
    const root = document.documentElement;
    if (theme === 'dark') root.setAttribute('data-theme', 'dark');
    else root.removeAttribute('data-theme');
    try {
      localStorage.setItem('admin:theme', theme);
    } catch {}
  }

  function initTheme() {
    const saved = getTheme();
    if (saved) setTheme(saved);
    else if (prefersDark()) setTheme('dark');
  }

  function toast(title, message) {
    const host = $('.toast-host');
    if (!host) return;
    const el = document.createElement('div');
    el.className = 'toast';
    el.innerHTML = `
      <div class="dot" aria-hidden="true"></div>
      <div>
        <div class="t"></div>
        <div class="m"></div>
      </div>
    `;
    el.querySelector('.t').textContent = title || 'Info';
    el.querySelector('.m').textContent = message || '';
    host.appendChild(el);
    // animate in
    requestAnimationFrame(() => el.classList.add('show'));
    // auto remove
    window.setTimeout(() => {
      el.classList.remove('show');
      window.setTimeout(() => el.remove(), 250);
    }, 2600);
  }

  function initSidebar() {
    const openBtn = $('[data-sidebar-open]');
    const closeBtn = $('[data-sidebar-close]');
    const overlay = $('[data-sidebar-overlay]');

    const open = () => document.body.classList.add('sidebar-open');
    const close = () => document.body.classList.remove('sidebar-open');

    openBtn?.addEventListener('click', open);
    closeBtn?.addEventListener('click', close);
    overlay?.addEventListener('click', close);

    document.addEventListener('keydown', (e) => {
      if (e.key === 'Escape') close();
    });
  }

  function initSearch() {
    const q = $('#search-q');
    const go = $('#search-go');
    if (!(q && go)) return;

    const navigate = () => {
      const v = (q.value || '').trim();
      const url = '/admin/search' + (v ? ('?q=' + encodeURIComponent(v)) : '');
      location.href = url;
    };

    go.addEventListener('click', navigate);
    q.addEventListener('keydown', (e) => {
      if (e.key === 'Enter') navigate();
      // Cmd/Ctrl + K focuses search
      if ((e.key === 'k' || e.key === 'K') && (e.metaKey || e.ctrlKey)) {
        e.preventDefault();
        q.focus();
      }
    });

    // Global shortcut: Cmd/Ctrl + K focuses search
    document.addEventListener('keydown', (e) => {
      if ((e.key === 'k' || e.key === 'K') && (e.metaKey || e.ctrlKey)) {
        e.preventDefault();
        q.focus();
      }
    });
  }

  function initThemeToggle() {
    const btn = $('[data-theme-toggle]');
    if (!btn) return;
    btn.addEventListener('click', () => {
      const current = document.documentElement.getAttribute('data-theme') === 'dark' ? 'dark' : 'light';
      const next = current === 'dark' ? 'light' : 'dark';
      setTheme(next);
      toast('Thema', next === 'dark' ? 'Donkere modus aan' : 'Lichte modus aan');
    });
  }

  async function copyText(text) {
    if (!text) return false;
    try {
      if (navigator.clipboard && navigator.clipboard.writeText) {
        await navigator.clipboard.writeText(text);
        return true;
      }
    } catch {}
    // Fallback
    try {
      const t = document.createElement('textarea');
      t.value = text;
      t.setAttribute('readonly', '');
      t.style.position = 'absolute';
      t.style.left = '-9999px';
      document.body.appendChild(t);
      t.select();
      const ok = document.execCommand('copy');
      t.remove();
      return ok;
    } catch {
      return false;
    }
  }

  function initCopy() {
    $$('[data-copy]').forEach((el) => {
      el.style.cursor = 'copy';
      el.title = el.title || 'Klik om te kopiëren';
      el.addEventListener('click', async () => {
        const text = el.getAttribute('data-copy') || el.textContent || '';
        const ok = await copyText(text.trim());
        toast(ok ? 'Gekopieerd' : 'Niet gelukt', ok ? text.trim() : 'Clipboard niet beschikbaar');
      });
    });
  }

  document.addEventListener('DOMContentLoaded', () => {
    initTheme();
    initSidebar();
    initSearch();
    initThemeToggle();
    initCopy();
  });
})();
